classdef DPP_cell < handle
    
    
    properties        
        spacer_1 = '------       Cell Source      ------';
        cell_path                   % path for for saving the cell
        shape_paths                 % source for the shape
        body_ind                    % body index from the labeled body matrix
        
        spacer_2 = '------     Shape Parameters   ------';
        % shape, polarity, yolk (mt stability-related)  matrices
        shape_mat               % shape 3d matrix; read from 'labbeled_bodies.mat'
        shape_center            % Shape center; Calculated from shape
        shape_span_avg        % Charcteristic size of the cell (radius; in pixels). Calculated from shape
        shape_span_max        % maximum of the spans size along [x, y, z].
        shape_span_x
        shape_span_y 
        shape_span_z

        spacer_3 = '--  Pulling Force and Polarity   ----';
        % Force equation : F = L^Beta1 + pullfactor*L^Beta2 + pullfactor_exp*2^(L/Unit_branch)
        polarity_mat
        Beta1
        pullfactor
        Beta2
        pullfactor_exp
        Unit_branch 
        Unit_branch_2       % DEDEBUG: unit branch hardcoded. Only Unit_branch_2 is used in the simulaitons.     

        
        % MT stability
        spacer_4 = '--------   MT. stability   ---------';        
        stab_mat
        yolk_mat
        yolk_show
        MTstab              % Robustness of MT in yolk. 1 is default.
        gradyolk            % sharpness of the yolk  %DEDEBUG: change names for yolk
        erfoffset           % offset of the error function
        direction_grad      % [theta, rad]        

        spacer_5 = '--------    Aster ''Size''   ---------';
                  
        anglelimit_deg          % MT body angle extent (counted from direciton of the spindle as 0)
        anglelimit_frac_of_180  % MT body angle extent as fraction of 180.
        
        spacer_6 = '-  Nucleus and Spindle Parameters  -';
        nucleus_d           % nucleus size = diameter
        nucleus_d_excl      % nucleus exclusion size (aka excluded volume). 
                            % minimum Nucexcl = 1; if 1 than excluded
                            % volume is the same as the normal size 
                            
        % simulation-VARIED parameters: subject to change during simulaiton
        x                   % int; nucleus center x; VARIES 
        y                   % int; nucleus center y; VARIES
        z                   % int; nucleus center z; VARIES
        theta               % spindle orientation; in degrees % VARIES
        phi                 % spindle orientation; in degrees % VARIES
    end    
     
     
     properties  (Hidden = true)
        % figure handles
        fig
        % axes children handles
        figAxCh_shape_patch        
        figAxCh_shape_center
        figAxCh_polarity_patch 
        figAxCh_nuclExcl_patch 
        figAxCh_nucl_patch
        figAxCh_nucl_center
        figAxCh_spindle_line       
        figAxCh_yolk_patch
        
         % figure proterties:
        col_shape           = [0 1 1]; %cyan
        col_yolk            = [1 0.5 0]; %orange
        col_polarity        = [1 0 0];
        col_shape_center    = [0 0 0.5]; 
        col_nucleus_surf    = [0 1 0];
        col_excl_nucleus_surf = [1 0 0];
        col_nucleus_center  = [0 0.5 0];
        col_spindle         = [0 0 0];
        
        nuclExcl_transp     = 0.1;
        nucl_transp         = 0.2;
        
        font_size           = 14;    
        
        yolk_more_transparency   = 0.3;  % transparency for yolk display.
                                    % 1 = no change (original yolk values
                                    % 0.5 = more transparent (half of the original yolk values).
                                    % in any case yolk concentration used
                                    % for MT stability concentration is not
                                    % changed.
     end
    
    
     % these parameters will vary depending on some simulaiton-varied parameters.
     properties (Dependent, Hidden = true)
         % nucleus: 
        nucleus_shape
        nucleus_shape_excl
        spindle_shape
     end
     
     
     
     
    
    methods
        
        function obj = DPP_cell(cell_params)
 
            disp(' ')
            disp(['Building a cell...'])
            disp(['    Source:'])
            disp(['    ' cell_params.shape_paths.shapes_source]) 
            disp(' ')

                        
            % POPULATE SOME PARAMETERS
            obj.shape_paths         = cell_params.shape_paths;
            obj.body_ind            = cell_params.body_ind;
            obj.cell_path           = DPP_cell.generate_cell_path(cell_params.shape_paths);            
            obj.fig                 = [];
            
            %   TRY TO GENERATE THE SHAPE MATRIX  
            % shape matrix is REQUIRED; if empty - program aborts.
            go_next  = try_to_make_the_shape(obj);
            if ~go_next
                return;
            end     
            
            %   TRY TO FIND THE SHAPE CENTER 
            try_to_find_the_shape_center(obj);
            
            %   TRY TO FIND THE CHARACTERISTIC SIZE 
            try_to_find_the_typical_size(obj);            

            
            %   NUCLEUS PARAMETERS
            % nucleus diameter and exclusion diameter (aka excluded volume)
            obj.nucleus_d           = cell_params.Nucsize;
            % exclusion volume parameter cannot be below 1:
            if cell_params.Nucexcl > 1
                obj.nucleus_d_excl  = cell_params.Nucexcl;
            else
                obj.nucleus_d_excl  = 1;
            end
            
            % position. This function is actually a SETTER for obj.x, obj.y, obj.z.
            % check inside.
            go_next                 = try_to_get_initial_nucleus_position(obj, cell_params);
            if ~go_next
                return;
            end
            
            
            % TRY TO GENERATE YOLK MATRIX     
            % MT spatial stability parameters
            obj.MTstab              = cell_params.MTstab;           % stability of MT in yolk;
            obj.gradyolk            = cell_params.gradyolk;         % sharpness of the yolk  %DEDEBUG: change names for yolk
            obj.erfoffset           = cell_params.erfoffset;        % offset of the error function
            obj.direction_grad      = cell_params.direction_grad;  	% [theta, rad] 
            
            % try to generate the yolk concentration matrix. All assignment
            % happens inside.
            try_to_generate_yolk_matrix(obj);            

                        
            
            %    TRY TO GENERATE POLARITY MATRIX  
            % Pulling force parameters.
            % Force equation : F = L^Beta1 + pullfactor*L^Beta2 + pullfactor_exp*2^(L/Unit_branch)
            obj.Beta1       	= cell_params.Beta1;
            obj.Beta2       	= cell_params.Beta2;
            obj.Unit_branch 	= cell_params.Unit_branch; 
            obj.pullfactor      = cell_params.pullfactor;
            obj.pullfactor_exp	= cell_params.pullfactor_exp;
            obj.Unit_branch_2 	= obj.Unit_branch / 45; 
            % DEDEBUG: unit branch is hardcoded. 
            % DEDEBUG: Only Unit_branch_2 is used in the Force calculator.
            
            % Polarity matrix is optional (may be empty = []).
            obj.polarity_mat        = try_to_make_the_polarity(obj); 
                      
                      
            
            % SET THE SPINDLE ORIENTATION:
            % according to AP2016:
            obj.theta = cell_params.theta;
            if cell_params.theta == 0
                obj.theta = 180;
            else                
            end   
            obj.phi = cell_params.phi;
            if cell_params.phi == 0
                obj.phi     = 180;
                obj.theta   = 180 - obj.theta;
            end     
                        

            obj.anglelimit_deg              = cell_params.anglelimit_deg;
            obj.anglelimit_frac_of_180      = cell_params.anglelimit_deg / 180; % angle as a fraction of 180 degrees (not rad)

        end
        
        
        
        function excl_nucleus_shape = get.nucleus_shape_excl(obj)
            % getter for obj.nucleus_shape_excl (which is dependend - meaning
            % that it does not exist: it gets calculated on the fly.
            
            % Structure is made for use with surf2patch as:
            % surf2patch(obj.nucleus_shape(:, :, 1), obj.nucleus_shape(:, :, 2), obj.nucleus_shape(:, :, 3));
            
            excl_nucleus_shape          = zeros(101, 101, 3);           
            [Xsphere, Ysphere, Zsphere] = sphere(100);
            
            radius_scale_factor         = (obj.nucleus_d / 2) * obj.nucleus_d_excl;
            
            excl_nucleus_shape(:,:,1)   = round(Xsphere * radius_scale_factor + obj.x);
            excl_nucleus_shape(:,:,2)   = round(Ysphere * radius_scale_factor + obj.y);
            excl_nucleus_shape(:,:,3)   = round(Zsphere * radius_scale_factor + obj.z); 
            
            
            if 0
                % DEDEBUG:
                % check if the exclusion is calculated properly.
            
                figure;
                hold on;
                
                % unit sphere
                % mesh(Xsphere, Ysphere, Zsphere) 
                
                 % excl nucl size
                plot(excl_nucleus_shape(:,:,1), excl_nucleus_shape(:,:,2), 'r+')
                % plot3(excl_nucleus_shape(:,:,1), excl_nucleus_shape(:,:,2), excl_nucleus_shape(:,:,3), 'ro')
                              
                % original nucl size
                normal_nucleus_shape = obj.nucleus_shape;
                plot(normal_nucleus_shape.x, normal_nucleus_shape.y, 'ko')
                %plot3(normal_nucleus_shape.x, normal_nucleus_shape.y, normal_nucleus_shape.z, 'k.')
            end
        end
        
        
        
        
        function normal_nucleus_shape = get.nucleus_shape(obj)
            % getter for obj.nucleus_shape (which is dependend - meaning
            % that it does not exist: it gets calculated on the fly.
            
            % Structure is made for use with surf2patch as:
            % surf2patch(obj.nucleus_shape.x, obj.nucleus_shape.y, obj.nucleus_shape.z);
            
            [Xsphere, Ysphere, Zsphere] = sphere(100);
            
            radius_scale_factor         = obj.nucleus_d / 2;

            normal_nucleus_shape        = [];
            normal_nucleus_shape.x      = round(Xsphere * radius_scale_factor + obj.x);
            normal_nucleus_shape.y      = round(Ysphere * radius_scale_factor + obj.y);
            normal_nucleus_shape.z      = round(Zsphere * radius_scale_factor + obj.z); 
        end
        
                
        
        function current_spindle_shape = get.spindle_shape(obj)
            % getter for obj.spindle_shape (which is dependend - meaning
            % that it does not exist: it gets calculated on the fly.
            
            current_spindle_shape       = [];
            current_spindle_shape.x     = []; % [x1 x2]
            current_spindle_shape.y     = []; % [y1 y2]
            current_spindle_shape.z     = []; % [z1 z2]
            
            % Structure is made for use with plot as:
            % phi_rad     = phi * pi/180;
            % theta_rad   = theta *pi/180;
            % plot3(current_spindle_shape.x, current_spindle_shape.y, current_spindle_shape.z,...
            %       '-k','LineWidth',3);
            
            phi_rad                 = obj.phi * pi/180;
            theta_rad               = obj.theta * pi/180;
            
            % Anaelle plotting was: 
            % x : [y+Nucsize*shape_span_avg/15/4*sin(phi_rad)*sin(theta_rad), y-Nucsize*shape_span_avg/15/4*sin(phi_rad)*sin(theta_rad)];
            % y : [x+Nucsize*shape_span_avg/15/4*cos(phi_rad)*sin(theta_rad),x-Nucsize*shape_span_avg/15/4*cos(phi_rad)*sin(theta_rad)];
            % z : [z+Nucsize*shape_span_avg/15/4*cos(theta_rad),z-Nucsize*shape_span_avg/15/4*cos(theta_rad)];
             
            % then according to Anaelle:
            current_spindle_shape.x = [-sin(phi_rad) * sin(theta_rad), sin(phi_rad) * sin(theta_rad)] * obj.nucleus_d/2 + obj.x;
            current_spindle_shape.y = [-cos(phi_rad) * sin(theta_rad), cos(phi_rad) * sin(theta_rad) ] * obj.nucleus_d/2 + obj.y;            
            current_spindle_shape.z = [-cos(theta_rad), cos(theta_rad)] * obj.nucleus_d/2 + obj.z;             
            
            % But she had [x,y,z] confused with [y,x,z]. So let's try to fix it:
            current_spindle_shape.x = [-cos(phi_rad) * sin(theta_rad), cos(phi_rad) * sin(theta_rad)] * obj.nucleus_d/2 + obj.x;
            current_spindle_shape.y = [-sin(phi_rad) * sin(theta_rad), sin(phi_rad) * sin(theta_rad)] * obj.nucleus_d/2 + obj.y;            
            current_spindle_shape.z = [-cos(theta_rad), cos(theta_rad)] * obj.nucleus_d/2 + obj.z;                       
        end
        
                
        
        
        function xyz_th_phi = get_state(obj)            
            % state = nucleus position (x,y,z) and spindle orientation (th, phi)
            xyz_th_phi    = [obj.x, obj.y, obj.z, obj.theta, obj.phi]; 
            
        end

         function set_state(obj, xyz_th_phi)            
            % state = nucleus position (x,y,z) and spindle orientation (th, phi)
            obj.x       = xyz_th_phi(1);
            obj.y       = xyz_th_phi(2);
            obj.z       = xyz_th_phi(3);
            obj.theta   = xyz_th_phi(4);
            obj.phi     = xyz_th_phi(5);
            disp('Current state: ')
            disp(['    x:     ' num2str(obj.x)])
            disp(['    y:     ' num2str(obj.y)])
            disp(['    z:     ' num2str(obj.z)])
            disp(['    theta: ' num2str(obj.theta)])
            disp(['    phi:   ' num2str(obj.phi)])            
         end

        
         
        % Check whether the nucleus is well inside the Shape
        function  inside_bool = check_if_nucleus_is_within_the_shape(obj)
             
            inside_bool = 1;
            
            ns      = obj.nucleus_shape_excl;            
            
            % find coordinates of x, y, z of the nucleus excluded shape
            all_x   = ns(:,:,1);	
            all_x   = all_x(:);
            all_y   = ns(:,:,2);
            all_y   = all_y(:);
            all_z   = ns(:,:,3);	
            all_z   = all_z(:);

            xyz     = [all_x, all_y, all_z];            
            xyz     = unique(xyz, 'rows');
            
            if any(xyz(:) < 1)
                inside_bool = 0;
                return;
            end
            
            % mesh(ns(:,:,1), ns(:,:,2), ns(:,:,3))
            % plot3(xyz(:,1), xyz(:,2), xyz(:,3), 'ro')
            
            % important: shape coordinates are [r,c,z], 
            % where r(ow) correspond to Y and c(ol) correspond to X;
            % thus we use Y coordinates as rows, X coordinates as cols. 
            for i = 1 : size(xyz, 1)
                if obj.shape_mat(xyz(i, 2), xyz(i, 1), xyz(i, 3)) == 0
                    inside_bool = 0;
                    break;
                end
            end
            
            
        end
        
        
        
        
        
        function make_cell_figure(obj)
            % Show the cell a surface;
  
            sz = get(0,'ScreenSize');
            figH = 0.8*sz(4);
            figW = figH;
            figY = 0.05*sz(4);
            figX = 0.1*sz(3);
            obj.fig = figure('position', [figX figY figW figH]);
            hold on;
            title({['cell shape source: '],...
                   [obj.shape_paths.folder_name, '\shape_source\labeled_bodies.mat'],...
                   ['Body label: ' num2str(obj.body_ind)]}, ...
                    'FontSize', obj.font_size, 'interpreter', 'none')

            axis equal            
            xlabel('X', 'FontSize', obj.font_size);
            ylabel('Y', 'FontSize', obj.font_size);
            zlabel('Z', 'FontSize', obj.font_size);
            xlim([0 size(obj.shape_mat, 2)]);
            ylim([0 size(obj.shape_mat, 1)]);
            zlim([0 size(obj.shape_mat, 3)]);

            box on;                
            view([-15, 15])  

            % shape
            obj.figAxCh_shape_patch  = patch(isosurface(obj.shape_mat, 0.5),...
                                        'FaceColor', obj.col_shape, 'FaceAlpha', 0.15, 'EdgeColor', 'none');

            % shape center            
            obj.figAxCh_shape_center = plot3(obj.shape_center(1), obj.shape_center(2), obj.shape_center(3),...
                                    'o', 'markerfacecolor', obj.col_shape_center, 'markeredgecolor', obj.col_shape_center, 'MarkerSize', 4);
          
            if 0
                % shape center projection (vertical axis)
                plot3([obj.shape_center(1), obj.shape_center(1)], ...
                    [obj.shape_center(2), obj.shape_center(2)],...
                    [1, size(obj.shape_mat, 3)], 'k-')
                plot3([obj.shape_center(1), obj.shape_center(1)], ...
                    [obj.shape_center(2), obj.shape_center(2)],...
                    [1, size(obj.shape_mat, 3)], ...
                    'o', 'markerfacecolor', obj.col_shape_center, 'markeredgecolor', obj.col_shape_center, 'MarkerSize', 4);
            end
            
            % polarity
            if ~isempty(obj.polarity_mat)
                obj.figAxCh_polarity_patch = patch(isosurface(obj.polarity_mat, 0.01),...
                                                'FaceColor', obj.col_polarity, 'FaceAlpha', 0.15, 'EdgeColor', 'none'); %, 'FaceAlpha', 0.1);
            end     


            % nucleus surface as excluded volume
            nucl_patch = surf2patch(obj.nucleus_shape_excl(:, :, 1), obj.nucleus_shape_excl(:, :, 2), obj.nucleus_shape_excl(:, :, 3));
            obj.figAxCh_nuclExcl_patch = patch(nucl_patch, 'FaceColor', obj.col_excl_nucleus_surf, 'FaceAlpha', obj.nuclExcl_transp, 'EdgeColor','none');

            % nucleus surface as normal volume
            nucl_patch = surf2patch(obj.nucleus_shape.x, obj.nucleus_shape.y, obj.nucleus_shape.z);
            obj.figAxCh_nucl_patch = patch(nucl_patch, 'FaceColor', obj.col_nucleus_surf, 'FaceAlpha', obj.nucl_transp, 'EdgeColor','none');

            % nucleus center
            obj.figAxCh_nucl_center = plot3(obj.x, obj.y, obj.z, ...
                                            'o', 'markerfacecolor', obj.col_nucleus_center, 'markeredgecolor', obj.col_nucleus_center, 'MarkerSize', 4);


            % spindle
            obj.figAxCh_spindle_line = plot3(obj.spindle_shape.x, obj.spindle_shape.y, obj.spindle_shape.z, ...
                                                '-', 'color', obj.col_spindle, 'linewidth', 2);
%             plot3([obj.shape_center(1), obj.shape_center(1)], ...
%                     [obj.shape_center(2), obj.shape_center(2)],...
%                     [1, size(obj.shape_mat, 3)], ...
%                     'o', 'markerfacecolor', obj.col_shape_center, 'markeredgecolor', obj.col_shape_center, 'MarkerSize', 4);
              



            % if there is no yolk, then everywhere inside the values are 1.
            % in this case  length(unique(obj.yolk_mat))  = 2.
            % so Only if there are more unique values, then we show the yolk.            
            if ~isempty(obj.yolk_show)
%                 
                    % modify the transparency a bit
                    transp_values               = obj.yolk_show.transparency;                    % original transparency as concentration
                    transp_values               = transp_values * obj.yolk_more_transparency;    % make it more transparent
                    transp_values(1)            = 1;                                        % Set only one face to fill the whole scale [0, 1].
                                                                                            % Otherwise it rescales ALL values automatically
                    
                    obj.figAxCh_yolk_patch = patch('Faces',     obj.yolk_show.faces,...
                                                   'Vertices',  obj.yolk_show.vertices, ...
                                                   'FaceColor', obj.col_yolk, 'EdgeColor', 'none');                                               
                    obj.figAxCh_yolk_patch.FaceVertexAlphaData = transp_values;            
                    obj.figAxCh_yolk_patch.FaceAlpha = 'flat';  
                %end
            end % yolk condition
            
            
        end
        
        
        
        
        
        function show(obj)
           

            % abort if no shape
            if isempty(obj.shape_mat)
                disp(['        ABORTING.'])
                disp(['        Reason: Shape matrix is empty.'])
                return;
            end
            
            % if creating for the first time 
            if isempty(obj.fig) 
                make_cell_figure(obj);
                drawnow
            else   
                % if the figure is not valid redo it.
                if ~isvalid(obj.fig)
                    make_cell_figure(obj);                    
                
                else
                    figure(obj.fig)
                    
                    % delete some handles and redraw them. This will make
                    % redrawing faster.
                    % It can be done even faster, if the data acessed directy
                    % I did this for lines.


                    % nucleus surface as excluded volume
                    delete(obj.figAxCh_nuclExcl_patch);
                    nucl_patch = surf2patch(obj.nucleus_shape_excl(:, :, 1), obj.nucleus_shape_excl(:, :, 2), obj.nucleus_shape_excl(:, :, 3));
                    obj.figAxCh_nuclExcl_patch = patch(nucl_patch, 'FaceColor', obj.col_nucleus_surf, 'FaceAlpha', obj.nuclExcl_transp, 'EdgeColor','none');

                    % nucleus surface as normal volume
                    delete(obj.figAxCh_nucl_patch);
                    nucl_patch = surf2patch(obj.nucleus_shape.x, obj.nucleus_shape.y, obj.nucleus_shape.z);
                    obj.figAxCh_nucl_patch = patch(nucl_patch, 'FaceColor', obj.col_nucleus_surf, 'FaceAlpha', obj.nucl_transp, 'EdgeColor','none');

                    % nucleus center
                    obj.figAxCh_nucl_center.XData = obj.x;
                    obj.figAxCh_nucl_center.YData = obj.y;
                    obj.figAxCh_nucl_center.ZData = obj.z;

                    % spindle.          
                    % recalculate the spindle chape
                    current_spindle_shape           = obj.spindle_shape;
                    % set the data
                    obj.figAxCh_spindle_line.XData  = current_spindle_shape.x;
                    obj.figAxCh_spindle_line.YData  = current_spindle_shape.y;
                    obj.figAxCh_spindle_line.ZData  = current_spindle_shape.z;

                    
                    
                    
                    drawnow
                end
            end
            
            if 0 
                fig_file_name = [obj.cell_path, 'cell_init.png'];
                disp(['        Saving as:']) 
                disp(['        ', fig_file_name])            
                print(gcf, '-dpng', fig_file_name);
                
                disp('        Done')
            end
        end      
        
        
        
        
      
        
        function clear_fig_handles(obj)
             % clear axes children handles; This is needed for saving the
             % data.
            obj.fig = [];
            obj.figAxCh_shape_patch = [];        
            obj.figAxCh_shape_center = [];
            obj.figAxCh_polarity_patch = []; 
            obj.figAxCh_nuclExcl_patch = []; 
            obj.figAxCh_nucl_patch = [];
            obj.figAxCh_nucl_center = [];
            obj.figAxCh_spindle_line = [];
            obj.figAxCh_yolk_patch = [];
        end
        
        
        
        
        
        
        function save(obj)
           disp('  ')
             if ~isempty(obj.shape_mat)
                 disp('    Saving the cell...')
             else
                disp(['    Saving.'])
                disp(['        ABORTING.'])                
                disp(['        Reason: Shape matrix is empty.'])                
                return                
             end  
            
            
            save_bool = 0;
            if ~exist(obj.cell_path, 'dir')
                st = mkdir(obj.cell_path);
                if st
                    disp('        Created new folder: ')
                    disp(['        ', obj.cell_path])
                    save_bool = 1;
                else
                    disp('        Could not create folder: ')
                    disp(['        ', curent_cell_path]) 
                end                
            end
           
            if save_bool                
              
                 % figure handles shall be emptied; otherwise - large file:
                 % disp('        Cell: clearing axes handles...')
                 obj.clear_fig_handles;                 
                 current_cell = obj;
                 
                 % saving
                 save([obj.cell_path, filesep, 'cell.mat'], 'current_cell')
                 disp(['        Done']) 
            else
                disp(['        CELL IS NOT SAVED.'])
            end            
        end
    end
    
    
    
    
     methods (Static)
         
         function cell_path  = generate_cell_path(shape_paths)             
             % first try to find if a "cell_n" file already exists
            root_path           = shape_paths.folder_abs_path;
            cur_ind             = 1;            
            folder_name         = @(n) ['cell_', num2str(n)];
            
            % if exists, update it to name_2, name_3 etc until a non-existing name is found.
            while exist([root_path, folder_name(cur_ind)], 'dir') 
                cur_ind = cur_ind + 1;
            end
            
            cell_path = [root_path, folder_name(cur_ind), filesep];
            
            disp('    Saving folder: ')
            disp(['    ' cell_path]); 
            disp('    ')
         end
         
         
     end
    
    
end







function go_next = try_to_make_the_shape(cur_cell)
% TRY TO FIND SHAPE IMAGES FOR THE INDICATED CELL. MAKE SHAPE MATRIX
% In shape images the cell is inside 1, outside 0.

shape_mat   = [];
go_next     = 0;

disp('    Making the Shape matrix.');

% try to load the labeled bodies data (if it exists). 
file_name           = 'labeled_bodies.mat';
file_path           = [cur_cell.shape_paths.shapes_source, file_name];
if ~exist(file_path, 'file')    
    disp(' ')
    disp(['Could not find the required file:']);
    disp(['    ', file_path]);
    disp(['    You need to generate this file.']);
    return;
end


disp(['        Reading labeled bodies...']);
load(file_path, 'all_bodies_labeled_uint8'); 
disp('        Done')


% change all elements to 0 except for those that are equal to the cell index
all_bodies_labeled_uint8(all_bodies_labeled_uint8 ~= cur_cell.body_ind) = 0;

shape_mat       = double(all_bodies_labeled_uint8);
% check shape:
% patch(isosurface(shape_mat, 0.5), 'FaceColor', [0 1 0], 'EdgeColor', 'none', 'FaceAlpha',0.15);


% Final checks (may be something went wrong during reading):
if isempty(shape_mat)
    disp(['        ABORTING.'])
    disp(['        Reason: Shape matrix is empty.'])                
    return
end       

if max(shape_mat(:)) == 0 % should be 0-1 matrix
    disp(['        ABORTING.'])
    disp(['        Reason: Shape matrix is not empty, but the indicated body is empty.'])
    disp(['        Indicated body index: ' num2str(cell_params.body_ind)])
    disp(['        Shape matrix is now set to empty'])
    shape_mat = [];
    return;
end
   
cur_cell.shape_mat = shape_mat;
go_next = 1;

end

   



function try_to_find_the_shape_center(curr_cell)
% FIND THE SHAPE CENTER OF MASS; [x,y,z]

disp('    Calculating the cell center...');
cc              = bwconncomp(curr_cell.shape_mat, 6);

if cc.NumObjects > 1
    disp(['        Warning!'])
    disp(['        It seems that there are more than one body.'])
    disp(['        It is recommended to clean the bodies from small junk better.']) 
    disp(['        Proceeding anyways: taking the largest body centroid.']) 
end
% you may check the shape:
% patch(isosurface(shape_mat,0.5));

% find all areas (=volume):
bb              = regionprops(cc, 'Centroid', 'Area');
areas           = cat(1, bb.Area);

% take the centroid from the largest area
xyz_centroid    = bb(areas == max(areas)).Centroid;
shape_center    = round(xyz_centroid);

curr_cell.shape_center        = shape_center;
disp('        Done');

 
end





function typical_size        = try_to_find_the_typical_size(curr_cell)
% find the principal sizes of the shape. Average of three axes.

disp('    Calculating the characteristic size...');
cc          = bwconncomp(curr_cell.shape_mat, 6);

if cc.NumObjects > 1
    disp(['        Warning!'])
    disp(['        It seems that there are more than one body.'])
    disp(['        It is recommended to clean the bodies from small junk better.']) 
    disp(['        Proceeding anyways: taking the largest span']) 
end
% you may check the shape:
% patch(isosurface(shape_mat,0.5));

% anyways, this works for any number of the bodies:
bb              = regionprops(cc, 'BoundingBox'); 
bb              = cat(1, bb.BoundingBox);
xyz_spans       = bb(:, 4:6);
mean_spans      = mean(xyz_spans, 2);
typical_size    = max(mean_spans);

curr_cell.shape_span_avg        = typical_size;
curr_cell.shape_span_x         = xyz_spans(1);
curr_cell.shape_span_y         = xyz_spans(2);
curr_cell.shape_span_z         = xyz_spans(3);
curr_cell.shape_span_max       = ceil(sqrt(xyz_spans(1)^2 + xyz_spans(2)^2 + xyz_spans(3)^2));

disp('        Done');



end






 
function go_next         = try_to_get_initial_nucleus_position(curr_cell, cell_params)
%   NUCLEUS POSITION

disp(['    Checking whether the nucleus is inside...'])

go_next = 0;

if length(cell_params.startingxyz) == 3
    curr_cell.x   = round(cell_params.startingxyz(1));
    curr_cell.y   = round(cell_params.startingxyz(2));
    curr_cell.z   = round(cell_params.startingxyz(3));
else
    curr_cell.x   = round(curr_cell.shape_center(1));
    curr_cell.y   = round(curr_cell.shape_center(2));
    curr_cell.z   = round(curr_cell.shape_center(3));
end

% check if the nucleus is inside the shape
nucl_inside_bool    = check_if_nucleus_is_within_the_shape(curr_cell);

if nucl_inside_bool
    go_next = 1;
    disp(['        Done'])
else
    disp(['        ABORTING.'])
    disp(['        Reason: the nucleus is outside the shape.'])
    go_next = 0;
end   



end




function polarity_mat = try_to_make_the_polarity(cur_cell)
% TRY TO FIND POLARITY IMAGES FOR THE INDICATED CELL. MAKE POLARITY MATRIX
% In polarity images the cell is inside 1, outside 0.

disp('    Making the polarity matrix...');

%   Polarity matrix by default is 0 everywhere inside the cell:            
polarity_mat           = zeros(size(cur_cell.shape_mat));  


if  cur_cell.pullfactor == 0 && cur_cell.pullfactor_exp == 0
    disp(['        Polarity matrix is set to 0.'])
    disp(['        Reason: Pulling factors are 0:'])
    disp(['        pullfactor = ' num2str(cur_cell.pullfactor)])
    disp(['        pullfactor_exp = ' num2str(cur_cell.pullfactor)])
    disp(['        If you want to define the polarity domain, set at least one of them to a value > 0.'])
    disp(['        Done (polarity matrix = 0)'])
    return;
end


% call a function for defining a polarity mat.
% define a global to hold the resulting polarity.
global global_polarity_cap;
global_polarity_cap = [];
set_polarity_cap_interactively(cur_cell)

polarity_mat = global_polarity_cap;
clear -global global_polarity_cap;


if  isempty(polarity_mat)
    disp(['        Polarity matrix is set to 0.'])
    disp(['        Reason: It was empty after interactive selection.'])
    disp(['        Done (polarity matrix = 0)'])
    
    % set everything to 0:
    polarity_mat            = zeros(size(cur_cell.shape_mat));  
    cur_cell.pullfactor     = 0;
    cur_cell.pullfactor_exp = 0;
    
else
    disp(['        Done (polarity matrix is set properly)'])
end



% CLOSING HOLES:
% warning('DE: modifyig polarity')
% % mask_z = imfill(Imagespol{z,1}, 'holes');
% img_dilated = imdilate(Imagespol{z,1}, strel('disk', 2));
% img_dilated_masked = img_dilated .* Imagesshape{z,1}; % using the shape as output
% Imagespol{z,1} = img_dilated_masked;
            

% OBSOLETE
% Normalize polarity matrix intensity
% warning('NORMALIZATION!!!')
%     max_intensity=max(max(max(Bsub)));    
%     if max_intensity>0
%         mult_int=100/max_intensity;
%     else mult_int=100;
%     end

end





function try_to_generate_yolk_matrix(cur_cell)
           
disp('    Making the MT stability matrix...');

gradyolk            = cur_cell.gradyolk;         % sharpness of the yolk  %DEDEBUG: change names for yolk
erfoffset           = cur_cell.erfoffset;        % offset of the error function
direction_grad      = cur_cell.direction_grad;   % [theta, rad]            
MTstab              = cur_cell.MTstab;           % MT stability 

% default values of Yolk  is Empty.
% default value of stab mat is 1 everywhere (no effect on length):
% L_eff(i) = L_0(i) * st(i) in each element (i):
cur_cell.yolk_show  = [];
cur_cell.stab_mat   = ones(size(cur_cell.shape_mat));


% check if the Yolk parameters  are provided properly:
if isempty(gradyolk)
    disp('        Yolk Concentration Gradient Parameters: ')
    disp('        Parameter gradyolk is empty.')
    disp('        MT stability is set to 1 everywhere inside the cell.');   
    disp('        If you want to define spatial gradient of MT stability, set gradyolk.');   
    disp('        Current gradyolk is set to 0:')    
    cur_cell.gradyolk  = 0;
    disp(['        gradyolk = ' num2str( cur_cell.gradyolk)])   
    return
end

% check if the Yolk parameters  are provided properly:
if ~(numel(gradyolk) == 1 && numel(direction_grad) == 2 && numel(erfoffset) == 1)
    disp('        Yolk Concentration Gradient Parameters: ')
    disp('        Input is ambiguous. Should be as:')
    disp('        gradyolk       = x, where x > 0')
    disp('        direction_grad = [theta, phi]')
    disp('        erfoffset      = y')
    disp('        MT stability is set to 1 everywhere inside the cell.');   
    disp('        Current gradyolk is set to 0:')
    cur_cell.gradyolk  = 0;
    return
end
    
if gradyolk <= 0
    disp('        Yolk Concentration Gradient Parameters: ')
    disp('        Parameter gradyolk should be higher than 0.')
    disp('        MT stability is set to 1 everywhere inside the cell.');   
    disp('        If you want to define spatial gradient of MT stability, set gradyolk > 0.');   
    disp('        Current gradyolk is set to 0:')
    cur_cell.gradyolk  = 0;
    disp(['        gradyolk = ' num2str( cur_cell.gradyolk)])   
    return
end

if gradyolk >0
    % carry on  
else
    % something weird.
    disp('        Yolk Concentration Gradient Parameters: ')
    disp('        Input is ambiguous. Should be as:')
    disp('        gradyolk       = x, where x > 0')
    disp('        direction_grad = [theta, phi]')
    disp('        erfoffset      = y')
    disp('        MT stability is set to 1 everywhere inside the cell.');   
    disp('        Current gradyolk is set to 0:')
    cur_cell.gradyolk  = 0;
    return
end


% if gradyolk is provided properly, carry on.
theta_deg           = direction_grad(1);
phi_deg             = direction_grad(2);
theta_rad           = theta_deg * pi/180;
phi_rad             = phi_deg * pi/180;

disp(['        Yolk Concentration Gradient Parameters: '])   
disp(['        Gradient direction, sharpness and offset:'])
disp(['        theta     = ' num2str(theta_deg), '; phi = ' num2str(phi_deg) '.']);
disp(['        gradyolk  = ' num2str(gradyolk)])
disp(['        offset    = ' num2str(erfoffset)])
disp(['        Calculating...'])


% DEDEBUG: check if x,y are not swapped in here
% vector unit in the direction of the gradient
% ATTENTION!!! COORDINATES ARE CONFUSING IN MATRICES!!!!
% U(ux,uy,uz) - unit vector; [ux is along x], [uy is y], [uz is z]
% BUT if one translates this vector to an image coordinates (2d matrix),
% they become U(uy, ux, uz) because of:
% x in an image = colums => 2nd dimension in the image matrix 
% y in an image = rows   => 1st dimension in the image matrix
% THUS when projecting a vector V(vx,vy,vz) which in matrix coordinates is (vy,vx,vz)
% on vector U(u1, u2, u3) we have to combine proper coordinates.

% ok, using standard trigonometry in cartesian coordinates:
unitvect_x        = sin(theta_rad) * cos(phi_rad);
unitvect_y        = sin(theta_rad) * sin(phi_rad);
unitvect_z        = cos(theta_rad);

% span of the gradient roughly:
if 0
    % LIMITED TO THE WHOLE MATRIX (According to AP2016):
    D               = ((size(cur_cell.shape_mat, 1) + size(cur_cell.shape_mat, 2) + size(cur_cell.shape_mat, 3) + 4) / 3);
    grad_center_x   = round(size(cur_cell.shape_mat, 2) / 2);
    grad_center_y   = round(size(cur_cell.shape_mat, 1) / 2);
    grad_center_z   = round(size(cur_cell.shape_mat, 3) / 2);
else
    % LIMITED TO THE CELL SHAPE ONLY
    % DE: It is better controlled:
    D               = cur_cell.shape_span_avg;
    grad_center_x   = cur_cell.shape_center(1);
    grad_center_y   = cur_cell.shape_center(2);
    grad_center_z   = cur_cell.shape_center(3);
end

disp(['            Yolk Concentration Matrix...'])

% initiate a gradient matrix with 0s and calculate the stability value at each x,y,z
% temporary holder
yolk_mat_temp    = zeros(size(cur_cell.shape_mat));


for y_ind   = 1 : size(cur_cell.shape_mat, 1)            % x = colums = 2nd dimension in the matrix. Now xgrad is NORMAL CARTESIAN X COMPONENT
    for x_ind = 1 : size(cur_cell.shape_mat, 2)          % y = rows = 1st dimension in the matrix. Now ygrad is NORMAL CARTESIAN Y COMPONENT
        for z_ind = 1 : size(cur_cell.shape_mat, 3)

            % distance of the projection of the point on gradient axis to center of image:
            % MAKE SURE WE BUILD THE VECTOR IN CARTESIAN COORDINATES BY
            % COMBINING PROPER CARTESIAN COMPONENTS:
            % projection of the point vector on the gradient vector:
            % (as a dot product)
            distgrad    = (x_ind - grad_center_x ) * unitvect_x + ... % xgrad - CARTESIAN X; size(shape_mat, 2) - 2nd matrix CARTESIAN X; unitvect(1) is CARTESIAN X.
                          (y_ind - grad_center_y ) * unitvect_y + ... % ygrad - CARTESIAN Y; size(shape_mat, 1) - 1st matrix CARTESIAN Y; unitvect(2) is CARTESIAN Y.
                          (z_ind - grad_center_z ) * unitvect_z;

            % ACCORDING TO THE AP2016 FORMULAS (two ways of calculation)
            % AP USED THE SECOND WAY IN FORCE CALCULATOR:
            % valuegrad   = 1/2 * (1 - erf((distgrad-erfoffset)  / S * gradyolk/2));
            % valuegrad2  = 1/2 * (1 - erf(-(distgrad-erfoffset) / S * gradyolk/2));

            % 1st way: with erf(+x)
            % yolk_conc_1  = 0.5 * (1 - erf(  gradyolk * (distgrad - erfoffset) / (2*D)) );
            % yolk_mat_1(y_ind, x_ind, z_ind) = valuegrad1; 
            
            % 2nd way: with erf(-x).            
            yolk_conc_2  = 0.5 * (1 - erf(- gradyolk * (distgrad - erfoffset) / (2*D)) );
            yolk_mat_temp(y_ind, x_ind, z_ind) = yolk_conc_2;  
            
        end
    end
end

% clip the Yolk concentration values to [0, 1]:
yolk_mat_temp(yolk_mat_temp < 0) = 0;
yolk_mat_temp(yolk_mat_temp > 1) = 1;

% mask the values that are outside the shape:
yolk_mat_temp       = cur_cell.shape_mat .* yolk_mat_temp;
cur_cell.yolk_mat   = yolk_mat_temp;
% yolk mat is done


%  --------------------   yolk display surface ------------------------
disp(['            Yolk Display Surface...'])

% erode the original shape, so that the majority of the
% isosurface would fall inside the cell:
shape_eroded        = imerode(cur_cell.shape_mat, strel('disk', 4));
yolk_mat_eroded     = yolk_mat_temp .* shape_eroded;   
[f, v]              = isosurface(yolk_mat_eroded, 0);
% v = [x, y, z]

if 0
    %DEDEBUG :check what is going on.
    figure;
    for kkk = 1:size(cur_cell.shape_mat, 3)
        cla;
        imagesc(cur_cell.shape_mat(:,:,kkk) + eroded_shape(:,:,kkk))
        pause(1/24)
        drawnow
    end
    
    
    % importantly, the order of v(1) and v(2)
    % coordinates.
    figure;
    imagesc(cur_cell.shape_mat(:, :, 50) + 10 * yolk_mat_eroded(:,:,50))

    v_i = 1500; % random vertex
    figure;
    imagesc(yolk_mat_temp(:, :, v(v_i, 3)))
    hold on;
    inds = v(:, 3) == v(v_i, 3);
    plot(v(inds, 1), v(inds, 2), 'r+')
end


% calculate the transparency values for faces.
% Use the yolk concentration as transparency.            

transp_values = NaN(size(f, 1), 1);
for i = 1 : size(f, 1)
    face_i_verts = v(f(i, :), :);
    transp_i = [];
    for j = 1:size(face_i_verts, 1)
        % IMPORTANT: 
        % v(1) = x = columns
        % v(2) = y = rows
        transp_i        = [transp_i,  yolk_mat_temp(face_i_verts(j, 2), face_i_verts(j, 1), face_i_verts(j, 3))];
    end
    transp_i            = mean(transp_i);
    transp_values(i)    = transp_i;
    % to check where the 0-yolk vertices are, one can set them to opaque:
    % if y_val==0
    %   transp_values(i) = 1;
    % end                
end

cur_cell.yolk_show.faces         = f;
cur_cell.yolk_show.vertices      = v;
cur_cell.yolk_show.transparency  = transp_values;




%  ----------------   calculating MT stability matrix   -------------------

disp(['            MT Stability Matrix...'])
sell_stab_mat       = 1 - yolk_mat_temp / MTstab;
sell_stab_mat       = cur_cell.shape_mat .* sell_stab_mat; % outside cell stability 0.
cur_cell.stab_mat   = sell_stab_mat;

disp('        Done')


% DEDEBUG
% if 0  
%    
%     % current point:
%     x_ind = shape_center_x;
%     y_ind = shape_center_y;
%     z_ind = shape_center_z - 20;
% 
%    figure;
%    for i = 1:size(yolk_mat_1, 3)
%       cla;
%       imagesc(yolk_mat_1(:, :, i))
%       title(num2str(i))
%       drawnow
%       waitforbuttonpress       
%    end
% end

   
end



